﻿using System;
using System.Linq;
using System.Web;
using System.Web.Mvc;

namespace NVCC.WebUI.Infrastructure
{
    public class NvccAuthorizeAttribute : AuthorizeAttribute
    {  
        private static IUserService UserService
        {
            get
            {
                return DependencyResolver.Current.GetService<IUserService>();
            }
        }

        protected override bool AuthorizeCore(HttpContextBase httpContext)
        {
            var isAuthenticated = false;
            var user = UserService.GetUser();

            if (user != null && user.ActiveFacilities().Count() > 0)
            {
                isAuthenticated = true;
            }
            return isAuthenticated;
        }

        protected override void HandleUnauthorizedRequest(AuthorizationContext filterContext)
        {
            filterContext.Controller.TempData["InfoMessage"] = "You must be authorized to a station to use that feature.";
            filterContext.Result = new RedirectToRouteResult(
                new System.Web.Routing.RouteValueDictionary {
                    { "controller", "User"},
                    { "action", "Login"},
                    { "returnController", filterContext.ActionDescriptor.ControllerDescriptor.ControllerName },
                    { "returnAction", filterContext.ActionDescriptor.ActionName }
            });
        }

        public override void OnAuthorization(AuthorizationContext filterContext)
        {
            if (filterContext == null)
            {
                throw new ArgumentNullException("filterContext");
            }

            if (AuthorizeCore(filterContext.HttpContext))
            {
                SetCachePolicy(filterContext);
            }
            else
            {
                HandleUnauthorizedRequest(filterContext);
            }
        }

        private void SetCachePolicy(AuthorizationContext filterContext)
        {
            // ** IMPORTANT **
            // Since we're performing authorization at the action level, 
            // the authorization code runs after the output caching module. 
            // In the worst case this could allow an authorized user 
            // to cause the page to be cached, then an unauthorized user would later 
            // be served the cached page. We work around this by telling proxies not to 
            // cache the sensitive page, then we hook our custom authorization code into 
            // the caching mechanism so that we have the final say on whether a page 
            // should be served from the cache.
            HttpCachePolicyBase cachePolicy = filterContext.HttpContext.Response.Cache;
            cachePolicy.SetProxyMaxAge(new TimeSpan(0));
            cachePolicy.AddValidationCallback(CacheValidationHandler, null /* data */);
        }

        private void CacheValidationHandler(HttpContext context,
                                            object data,
                                            ref HttpValidationStatus validationStatus)
        {
            validationStatus = OnCacheAuthorization(new HttpContextWrapper(context));
        }
    }
}